using System.Collections.Generic;
using UnityEngine;

/*
 * Handler Scripts to Control Empty Voter Objects used in Non-Visualised LD Simualtions (Non-visualised version of VoterHandler.cs)
 */
public class VoterScript : MonoBehaviour
{
    public int voterID;
    double competence = 0.5;
    public double percievedCompetence;
    float delegationChance = 95;

    int vote;
    int correctVotes;
    int incorrectVotes;

    Dictionary<int, VoterScript> localNetwork = new Dictionary<int, VoterScript>();
    bool delegates = false;
    float alpha = 0.02f;
    VoterScript delegatesTo;
    VoterScript guru;
    VoterScript prevGuru;
    bool reviewed;

    float preferentialAttachment;
    float competenceImportance;

    Dictionary<int, VoterScript> delegators = new Dictionary<int, VoterScript>();
    Dictionary<int, VoterScript> directDelegators = new Dictionary<int, VoterScript>();

    // Start is called before the first frame update, sets default values for voter
    void Start()
    {
        guru = this;
        prevGuru = this;

        correctVotes = 0;
        incorrectVotes = 0;
    }

    // Sets up voter agent parameters with user settings and generated numbers
    public void SetupVoter(int voterID, double addCompetence, float speed, float prefAttach, float compImp, float alphaValue)
    {
        this.voterID = voterID;

        competence += (addCompetence / 100);
        if (competence > 0.95f) { competence = 0.95f; }

        if (StatsManager.instance.roundType == 0)
        {
            percievedCompetence = competence;
        }
        else
        {
            percievedCompetence = 0.5;
        }

        preferentialAttachment = prefAttach;
        competenceImportance = compImp;
        alpha = alphaValue;

        if (!StatsManager.instance.setChance)
        {
            if (competence * 100 >= 51)
            {
                delegationChance = 95 - ((float)System.Math.Log((competence * 100) - 50) / (float)System.Math.Log(45) * 90);
            };
        }
        else
        {
            delegationChance = 50;
        }
    }

    // Local network of voters the voter knows added into dictionary
    public void SetupNetwork(int networkSize, VoterScript[] voterNetwork)
    {
        int netStart;
        int netEnd;

        if (networkSize % 2 == 0)
        {
            if (Random.Range(0, 2) == 0)
            {
                netStart = voterID - networkSize / 2;
                netEnd = voterID + (networkSize / 2 - 1);
            }
            else
            {
                netStart = voterID - (networkSize / 2 - 1);
                netEnd = voterID + (networkSize / 2);
            }
        }
        else
        {
            netStart = voterID - (networkSize - 1) / 2;
            netEnd = voterID + (networkSize - 1) / 2;
        }
        if (netStart < 0)
        {
            netEnd += Mathf.Abs(netStart);
            netStart = 0;
        }
        else if (netEnd >= StatsManager.instance.numVoters)
        {
            netStart -= (netEnd + 1 - StatsManager.instance.numVoters);
            netEnd = StatsManager.instance.numVoters - 1;
        }

        for (int i = netStart; i <= netEnd; i++)
        {
            if (voterNetwork[i].voterID != voterID)
            {
                localNetwork.Add(voterNetwork[i].voterID, voterNetwork[i]);
            }
        }
    }

    
    // Voter decides whether to be a guru or delegator based on delegation chance
    public void GuruOrDelegator(VoterScript[] voterNetwork)
    {
        int isGuru; 
        int guruChance = Random.Range(1, 101);
        if (guruChance > delegationChance) { isGuru = 1; } 
        else { isGuru = 0; }

        if (isGuru == 0)
        {
            delegates = true;

            int numHigherComp = 0;
            while (voterNetwork[numHigherComp].percievedCompetence >= competence + alpha)
            {
                numHigherComp++;
            }

            int numAccDel = 0;
            for (int i = 0; i < numHigherComp; i++)
            {
                if (localNetwork.ContainsKey(voterNetwork[i].voterID)
                    && voterNetwork[i].guru.voterID != prevGuru.voterID && !delegators.ContainsKey(voterNetwork[i].voterID))
                {
                    numAccDel++;
                }
            }

            if (numAccDel == 0) { isGuru = 1; }
            else
            {
                VoterScript[] accDelegations = new VoterScript[numAccDel];
                int i = 0;
                for (int j = 0; j < numHigherComp; j++)
                {
                    if (localNetwork.ContainsKey(voterNetwork[j].voterID)
                        && voterNetwork[j].guru.voterID != prevGuru.voterID && !delegators.ContainsKey(voterNetwork[j].voterID))
                    {
                        accDelegations[i] = voterNetwork[j];
                        i++;
                        if (i >= numAccDel) { break; }
                    }
                }

                GiveDelegation(numAccDel, accDelegations);
            }
        }

        if (isGuru == 1)
        {
            //print("Voter " + voterID + " with competence: " + competence + " is a Guru!");
            delegates = false;
        }
    }

    // Voter delegates their vote to another voter
    public void GiveDelegation(int numAccDel, VoterScript[] accDelegations)
    {
        int totalWeight = 0;
        int[] powerWeights = new int[numAccDel];
        for (int i = 0; i < numAccDel; i++)
        {
            int prefPower = (int)Mathf.Pow(1 + accDelegations[i].delegators.Count, preferentialAttachment);
            int compPower = (int)Mathf.Pow(1 + (100 * ((float)accDelegations[i].percievedCompetence - ((float)competence + alpha))), competenceImportance);

            int power = prefPower + compPower;
            powerWeights[i] = totalWeight + power;

            totalWeight += power;
        }

        int delegation = 0;
        int randomChoice = Random.Range(0, totalWeight);
        foreach (int powerWeight in powerWeights)
        {
            if (randomChoice <= powerWeight)
            {
                break;
            }
            delegation++;
        }

        delegatesTo = accDelegations[delegation];
        accDelegations[delegation].RecieveDelegation(this, true);
    }

    // Voter removes their current delegation
    public void RemoveDelegation()
    {
        delegates = false;
        prevGuru = guru;
        guru = this;

        delegatesTo.LoseDelegation(this, true);
        delegatesTo = null;
    }

    // Voter receives a delegation from another voter
    public void RecieveDelegation(VoterScript delegator, bool direct)
    {
        if (!directDelegators.ContainsKey(delegator.voterID) && direct)
        {
            directDelegators.Add(delegator.voterID, delegator);
        }

        if (!delegators.ContainsKey(delegator.voterID))
        {
            delegators.Add(delegator.voterID, delegator);
            delegator.SetGuru(this);
        }

        foreach (VoterScript subDelegator in delegator.delegators.Values)
        {
            if (!delegators.ContainsKey(subDelegator.voterID))
            {
                delegators.Add(subDelegator.voterID, subDelegator);
                subDelegator.SetGuru(this);
            }
        }

        if (delegates)
        {
            delegatesTo.RecieveDelegation(delegator, false);
        }
    }

    // Voter loses a delegation from another voter
    public void LoseDelegation(VoterScript delegator, bool direct)
    {
        if (direct)
        {
            directDelegators.Remove(delegator.voterID);
        }

        delegators.Remove(delegator.voterID);

        foreach (VoterScript subDelegator in delegator.delegators.Values)
        {
            if (delegators.ContainsKey(subDelegator.voterID))
            {
                delegators.Remove(subDelegator.voterID);
                subDelegator.SetGuru(delegator);
                subDelegator.prevGuru = this;
            }
        }

        if (delegates)
        {
            delegatesTo.LoseDelegation(delegator, false);
        }
    }

    // Voter votes, makes right decision based on competence, only used by Gurus
    public void CastVote(out int vote, out int power)
    {
        int decision = Random.Range(1, 101);
        if (competence * 100 >= decision) 
        { 
            vote = 1;
            correctVotes++;
        }
        else 
        { 
            vote = 0;
            incorrectVotes++;
        }
        this.vote = vote;
        reviewed = false;

        power = 1 + delegators.Values.Count;

        foreach (VoterScript delegator in delegators.Values)
        {
            delegator.GuruVote(vote);
        }
    }

    // Recieve vote option made by guru
    public void GuruVote(int vote)
    {
        this.vote = vote;
        reviewed = false;
    }

    // Returns if voter is a guru
    public bool IsGuru()
    {
        return !delegates;
    }

    // Return competence of voter
    public double GetTrueCompetence()
    {
        return competence;
    }

    // Perceived competence of voter updated based on how many correct votes they have made
    public void UpdatePercievedComp()
    {
        percievedCompetence = (double)correctVotes / (correctVotes + incorrectVotes);
        if (percievedCompetence < 0.5) { percievedCompetence = 0.5; }
    }

    // Guru of voter is set
    public void SetGuru(VoterScript guru)
    {
        this.guru = guru;

        if (!localNetwork.ContainsKey(guru.voterID))
        {
            localNetwork.Add(guru.voterID, guru);
        }
    }

    // Voter checks if they made the right decision if they were a guru last round
    public void ReviewVote(VoterScript[] votersByComp)
    {
        if (!reviewed)
        {
            reviewed = true;
            if (vote == 0)
            {
                VoterScript[] potentialyLostDelegates = new VoterScript[directDelegators.Count];
                directDelegators.Values.CopyTo(potentialyLostDelegates, 0);

                foreach (VoterScript voter in potentialyLostDelegates)
                {
                    voter.ReviewDelegation(votersByComp);
                }

                GuruOrDelegator(votersByComp);
            }
        }
    }

    // Voter checks if their guru made the right decision if they were a delegator last round
    public void ReviewDelegation(VoterScript[] votersByComp)
    {
        if (!reviewed)
        {
            reviewed = true;

            if (vote == 0)
            {
                double changeChance = 50 - 100 * (guru.percievedCompetence - competence);
                int changeChoice = Random.Range(1, 101);

                if (changeChance >= changeChoice)
                {
                    RemoveDelegation();
                    GuruOrDelegator(votersByComp);
                }
                else if (directDelegators.Count > 0)
                {
                    VoterScript[] potentialyLostDelegates = new VoterScript[directDelegators.Count];
                    directDelegators.Values.CopyTo(potentialyLostDelegates, 0);

                    foreach (VoterScript voter in potentialyLostDelegates)
                    {
                        voter.ReviewDelegation(votersByComp);
                    }
                }
            }
        }
    }

    // Voter removed from simulation
    public void Remove()
    {
        Destroy(gameObject);
        Destroy(this);
    }
}